<?php

namespace App\Livewire\Admin\Motors;

use App\Models\Motor;
use App\Models\MotorBrand;
use Livewire\Component;
use Livewire\WithFileUploads;

class Create extends Component
{
    use WithFileUploads;

    public $name;
    public $description;
    public $brand_id;
    public $cc;
    public $transmission = 'Automatic';
    public $fuel_type = 'Petrol';
    public $price_daily;
    public $price_weekly;
    public $price_monthly;
    public $main_image;
    public $images = []; // Multi-image gallery
    public $status = 'available';

    public function rules()
    {
        return [
            'name' => 'required|string|max:255',
            'description' => 'required|string',
            'brand_id' => 'required|exists:motor_brands,id',
            'cc' => 'required|integer|min:50|max:2000',
            'transmission' => 'required|in:Automatic,Manual',
            'fuel_type' => 'required|string',
            'price_daily' => 'required|numeric|min:0',
            'price_weekly' => 'nullable|numeric|min:0',
            'price_monthly' => 'nullable|numeric|min:0',
            'main_image' => 'required|image|max:10240', // 10MB Max
            'images.*' => 'image|max:10240', // Gallery validation
            'status' => 'required|in:available,booked,maintenance',
        ];
    }

    public function save()
    {
        $validatedData = $this->validate();

        if ($this->main_image) {
            $validatedData['main_image'] = $this->main_image->store('motors', 'public');
        }

        // Handle Gallery
        if ($this->images) {
            $galleryPaths = [];
            foreach ($this->images as $image) {
                $galleryPaths[] = $image->store('motors/gallery', 'public');
            }
            $validatedData['images'] = $galleryPaths;
        }

        Motor::create($validatedData);

        session()->flash('success', 'Motorbike created successfully.');

        return redirect()->route('admin.motors.index');
    }

    public function render()
    {
        return view('livewire.admin.motors.create', [
            'brands' => MotorBrand::all()
        ])->layout('layouts.admin');
    }
}
