<?php

namespace App\Livewire\Admin\Motors;

use App\Models\Motor;
use App\Models\MotorBrand;
use Livewire\Component;
use Livewire\WithFileUploads;

class Edit extends Component
{
    use WithFileUploads;

    public Motor $motor;

    public $name;
    public $description;
    public $brand_id;
    public $cc;
    public $transmission;
    public $fuel_type;
    public $price_daily;
    public $price_weekly;
    public $price_monthly;
    public $new_main_image;
    public $images = []; // Existing gallery
    public $new_images = []; // New uploads
    public $status;

    public function mount(Motor $motor)
    {
        $this->motor = $motor;
        $this->name = $motor->name;
        $this->description = $motor->description;
        $this->brand_id = $motor->brand_id;
        $this->cc = $motor->cc;
        $this->transmission = $motor->transmission;
        $this->fuel_type = $motor->fuel_type;
        $this->price_daily = $motor->price_daily;
        $this->price_weekly = $motor->price_weekly;
        $this->price_monthly = $motor->price_monthly;
        $this->images = $motor->images ?? [];
        $this->status = $motor->status;
    }

    public function rules()
    {
        return [
            'name' => 'required|string|max:255',
            'description' => 'required|string',
            'brand_id' => 'required|exists:motor_brands,id',
            'cc' => 'required|integer|min:50|max:2000',
            'transmission' => 'required|in:Automatic,Manual',
            'fuel_type' => 'required|string',
            'price_daily' => 'required|numeric|min:0',
            'price_weekly' => 'nullable|numeric|min:0',
            'price_monthly' => 'nullable|numeric|min:0',
            'new_main_image' => 'nullable|image|max:10240',
            'new_images.*' => 'image|max:10240',
            'status' => 'required|in:available,booked,maintenance',
        ];
    }

    public function removeImage($index)
    {
        if (isset($this->images[$index])) {
            unset($this->images[$index]);
            $this->images = array_values($this->images); // Reindex
        }
    }

    public function save()
    {
        $validatedData = $this->validate();

        if ($this->new_main_image) {
            $validatedData['main_image'] = $this->new_main_image->store('motors', 'public');
        } else {
            $validatedData['main_image'] = $this->motor->main_image;
        }

        // Handle Gallery
        $currentImages = $this->images; // Start with existing (after deletions)
        
        if ($this->new_images) {
            foreach ($this->new_images as $image) {
                $currentImages[] = $image->store('motors/gallery', 'public');
            }
        }
        
        $validatedData['images'] = $currentImages;

        unset($validatedData['new_main_image']);
        unset($validatedData['new_images']);

        $this->motor->update($validatedData);

        session()->flash('success', 'Motorbike updated successfully.');

        return redirect()->route('admin.motors.index');
    }

    public function render()
    {
        return view('livewire.admin.motors.edit', [
            'brands' => MotorBrand::all()
        ])->layout('layouts.admin');
    }
}
